package mcfall.math;

public abstract class Vector extends Matrix {
	
	/**
	 * Constructor a Vector with the specified number of rows and columns.  
	 * @param numRows the number of rows for a column vector, or 1 for a row vector
	 * @param numColumns the number of columns for a row vector, or 1 for a column vector
	 * @throws IllegalArgumentException since a Vector must have at least 1 of its 
	 * two dimensions have size 1, this constructor throws an IllegalArgumentException if neither value is 1
	 */
	protected Vector (int numRows, int numColumns) throws IllegalArgumentException {
		this (numRows, numColumns, null);
	}
	
	/**
	 * Constructor a Vector with the specified number of rows and columns, and the specified initial values.  
	 * @param numRows the number of rows for a column vector, or 1 for a row vector
	 * @param numColumns the number of columns for a row vector, or 1 for a column vector
	 * @throws IllegalArgumentException since a Vector must have at least 1 of its 
	 * two dimensions have size 1, this constructor throws an IllegalArgumentException if neither value is 1
	 */
	protected Vector(int numRows, int numColumns, double[] initialValues) throws IllegalArgumentException {
		super (numRows, numColumns, createInitialValuesForConstructor(numRows, numColumns, initialValues));				
	}

	/*
	 * Sets the value of the i'th component to the specified value
	 * @param i the index of the component to change
	 * @param value the new value for the component
	 * @return returns the old value stored at component <i>i</i>
	 * @throws IndexOutOfBoundsException if it is <i>i</i> is not between getFirstColumnIndex()/getFirstRowIndex() and getLastColumnIndex()/getLastRowIndex()
	 */
	public abstract double setValueAt (int i, double value) throws IndexOutOfBoundsException;
	
	/**
	 * Returns Vector that has the same dimensions and data values as this Vector
	 * @return a new Vector that is a copy of this Vector
	 */
	public abstract Vector duplicate ();
	
	/*
	 * Gets the value of the i'th component of this vector
	 * @param i the index of the component to retrieve
	 * @return returns the value stored at component <i>i</i>
	 * @throws IndexOutOfBoundsException if it is <i>i</i> is not between getFirstColumnIndex()/getFirstRowIndex() and getLastColumnIndex()/getLastRowIndex()
	 */
	public abstract double getValueAt (int i) throws IndexOutOfBoundsException;
		
	/**
	 * Computes the index of the first component of this vector
	 * @return an integer representing the index of the first component of this vector
	 */
	public abstract int getFirstIndex ();
	
	/**
	 * Computes the number of components of the vector
	 * @return an integer indicating the number of components in this vector
	 */
	public abstract int getSize ();
	/**
	 * Calculates the length of the Vector
	 * @return a double value indicating the length of the vector
	 */
	public double length () {
		try {
			return Math.sqrt (this.dot(this));
		}
		catch (IncompatibleMatrixException impossible) {
			System.err.println ("Impossible incompatible matrix exception thrown dotting a vector with itself");
			return -1;
		}
	}
	
	/**
	 * Creates a Vector which is the reflection of this Vector about the specified
	 * normal vector
	 * @param normal the normal vector to reflect about, which is assumed to
	 * be a unit vector
	 * @return a new Vector of the same length as this vector 
	 */
	// TODO figure out the difference between this and reflectNormal and y substituting one for the other doesnt work
	// TODO figure out y this works at all
	public Vector reflect (Vector normal)  {
		try {
			Vector negS = Vector.fromColumnMatrix(this.scalarMultiply(-1.0));
			
			//  Compute the vector of reflection
			double multiplier = 2*this.dot(normal)/Math.pow(normal.length(),2);
			Vector product = Vector.fromColumnMatrix(normal.scalarMultiply(multiplier));
			//  Add this product to negS
			Vector r = new ColumnVector (4);
			for (int i = 1; i < 4; i++) {
				r.setValueAt(i, negS.getValueAt(i) + product.getValueAt(i));
		}	
		return r;
		}
		catch (IncompatibleMatrixException badMatrixComputation) {
			throw new RuntimeException ("An unexpected exception occurred performing a reflect operation", badMatrixComputation);
		}
	}
	public boolean isBehindPlane(Vector normal) {
		try {
			return this.dot(normal)>0d;
		} catch (IncompatibleMatrixException e) {
			throw new RuntimeException ("An unexpected exception occurred performing an isBehindPlane operation", e);
		}
	}
	//TODO is this the same as above?
	public Vector reflectNormal (Vector normal)  {
		try {
			normal = normal.normalize();
			double dotp = this.dot(normal);
			Vector r = new ColumnVector(4);
			r = Vector.fromColumnMatrix(normal.scalarMultiply(2.0d*dotp));
			r = this.subtract(r); //ray - (-2*(this dot n)*n)
			return r;
		}
		catch (IncompatibleMatrixException badMatrixComputation) {
			throw new RuntimeException ("An unexpected exception occurred performing a reflect operation", badMatrixComputation);
		}
	}
	// TODO check this there appears to be a small error...a refractive index of one should be tranceparency
	/**
	 * Computes the refracted ray based on the refraction index and normal based on a modification of snells law which is:
	 * r = refraction*i - (refraction*i.n + sqrt(1-refraction*refraction*(1-i.n*i.n)))*n
	 * where r is the output vector i is the incident vector, n is the normal vector (which must be on the same side of the plane as the incident vector)
	 * Note:This function assumes that if your normal is initially on the opposite side of the plane as the incident then the incident ray must be coming from inside the object and therefore the refraction=1/refraction
	 * @param normal the normal
	 * @param refraction the ratio of the refractive indecies between the starting medium and the ending medium
	 * 
	 * @return the vector
	 */
	public Vector refract(Vector normal, double refraction) {
		try {
			normal = normal.normalize(); //make sure we are normalized
			Vector incident = this.normalize();
			final double cos = normal.dot(incident);
			if(cos>0)//this ray is <90 deg. from normal indicating we are comming from the inside (assumes vectors face outwards on a surface) 
			{
				refraction = 1d/refraction;//invert the refraction ratio
				normal = Vector.fromColumnMatrix(normal.scalarMultiply(-1d));//change the normal to be on the same side since the equation depends on it
			}
			Vector incidentComponent = Vector.fromColumnMatrix(incident.scalarMultiply(refraction));
			final double sin2 = (1.0 - cos*cos);
			Vector normalComponent = Vector.fromColumnMatrix(normal.scalarMultiply(refraction*cos + Math.sqrt(1d-refraction*refraction*sin2)));
			return incidentComponent.subtract(normalComponent);
		} catch (IncompatibleMatrixException e) {
			throw new RuntimeException ("An unexpected exception occurred performing a refract operation", e);
		}
		
	}
	public Vector add(Vector v) 
	{
		ColumnVector result = new ColumnVector(4);
		for(int i = 1;i<4;i++) 
		{
			result.setValueAt(i, this.getValueAt(i)+v.getValueAt(i));
		}
		return result;
	}
	public Vector subtract(Vector v) 
	{
		ColumnVector result = new ColumnVector(4);
		for(int i = 1;i<4;i++) 
		{
			result.setValueAt(i, this.getValueAt(i)-v.getValueAt(i));
		}
		return result;
	}
	/**
	 * Calculates the mathematical dot product
	 * @param other a vector of the same size as this vector
	 * @return the dot product of this vector with the vector <i>other</i> 
	 * @throws IncompatibleMatrixException if this vector and the vector <i>other</i> are not of the
	 * same size
	 */
	public double dot (Vector other) {
		Matrix returnValue;
		
		//  We want to ensure that we have a 1xn matrix multiplied by an nx1 matrix so we get a 1x1 result
		if (getNumberOfRows() > 1) {
			returnValue = premultiply(other.transpose());
		}
		else {
			returnValue = postmultiply(other.transpose());
		}
		
		return returnValue.getValueAt(getFirstRowIndex(), getFirstColumnIndex());
	}
	
	//  TODO:  Write description of cross product method
	public Vector cross (Vector right) {
		if (this.getSize() != 4 || right.getSize() != 4) {
			throw new IncompatibleMatrixException (this, right);
		}
		int ourFirstIndex = 1;
		int rightFirstIndex = 1;
		
		//  this.y*right.z - this.z*right.y, this.z*right.x-this.x*right.z, this.x*other.y-this.y*other.x
		double[] initialValues = new double[4];
		initialValues[0] = getValueAt (ourFirstIndex+1)*right.getValueAt(rightFirstIndex+2) - getValueAt(ourFirstIndex+2)*right.getValueAt(rightFirstIndex+1);
		initialValues[1] = getValueAt (ourFirstIndex+2)*right.getValueAt(rightFirstIndex) - getValueAt(ourFirstIndex) * right.getValueAt(rightFirstIndex+2);
		initialValues[2] = getValueAt (ourFirstIndex) * right.getValueAt(rightFirstIndex+1) - getValueAt(ourFirstIndex+1) * right.getValueAt(rightFirstIndex);
		return new ColumnVector (4, initialValues);				
	}
	
	/**
	 * Create a vector in the same direction as this vector with a length of 1
	 * @return a <b>new</b> unit vector in the same direction as this vector 
	 */
	public Vector normalize () {
		double length = length ();
		if(length!=1.0d) {//should make things a bit faster if we are already normalized
			Vector returnVector = duplicate ();
			for (int row = returnVector.getFirstRowIndex(); row <= returnVector.getLastRowIndex(); row++) {
				for (int col = returnVector.getFirstColumnIndex(); col <= returnVector.getLastColumnIndex(); col++) {
					returnVector.setValueAt(row, col, returnVector.getValueAt(row, col)/length);
				}
			}
			return returnVector;
		} else {
			return this;
		}
		
	}
	
	/**
	 * Computes the cosine of the angle between this vector and the vector <i>other</i>
	 * @param other a vector of the same size as this vector
	 * @return the cosine of the angle between this vector and the vector <i>other</i>, in the
	 * range -1 to 1
	 * @throws IncompatibleMatrixException if these vectors are not of the same size
	 */
	public double angleBetweenCosine (Vector other) {		
		return normalize().dot(other.normalize());
	}
	
	
	private static double[][] createInitialValuesForConstructor (int numRows, int numColumns, double[] initialValues) {
		if (numRows != 1 && numColumns != 1) {
			throw new IllegalArgumentException ("At least one of numRows, numColumns must be 1 for a Vector");
		}
		
		if (initialValues == null) {
			return null;
		}
		
		double[][] values;
		
		//  numRows = 1 means all values need to go in a single array
		if (numRows == 1) {
			values = new double[1][initialValues.length];
			System.arraycopy(initialValues, 0, values[0], 0, initialValues.length);
		}
		else {
			values = new double[initialValues.length][1];
			for (int i = 0; i < initialValues.length; i++) {
				values[i][0] = initialValues[i];
			}
		}
		
		return values;
	}

	/**
	 * Creates a ColumnVector from the given matrix, which is expected to be 
	 * a column matrix.  This method is a helper when matrix operations
	 * like pre/post multiply return matrices when we want to treat the 
	 * result as a Vector
	 * @param matrix
	 * @return a new Vector object with 4 rows.  The x coordinate is the entry in the first row of <i>matrix</i>,
	 * the y coordinate is the entry in the second row of <i>matrix</i>, and zthe  coordinate is the third row
	 * of <i>matrix</i>.  The final row, in keeping with homogeneous coordinates, has a value of 0
	 */
	public static Vector fromColumnMatrix(Matrix matrix) {				
		if (matrix.getNumberOfColumns() != 1) {
			throw new IncompatibleMatrixException (matrix, null);
		}
		int firstRowIndex = matrix.getFirstRowIndex();
		int firstColumnIndex = matrix.getFirstColumnIndex();
		double[] initialValues = new double[4];
		
		initialValues[0]= matrix.getValueAt(firstRowIndex, firstColumnIndex);
		initialValues[1] = matrix.getValueAt(firstRowIndex+1, firstColumnIndex);
		initialValues[2] = matrix.getValueAt(firstRowIndex+2, firstColumnIndex);
		initialValues[3] = 0.0;		 
		return new ColumnVector (4, initialValues);						
	}
	
	/**
	 * Determines whether this Vector is equal to another one
	 * Two vectors are equal if they have the same size, and each element has the same value
	 * They do not have to share the same starting index value, nor do they have to be the
	 * same type (i.e., a row vector can be considered to be equal to a column vector)
	 * @return true if <i>o</i> is a vector and this vector should be considered equal to <i>o</i>,
	 * false otherwise
	 */
	public boolean equals (Object o) {
		if (!(o instanceof Vector)) {
			return false;
		}
		
		Vector other = (Vector) o;
		if (other.getSize() != getSize()) {
			return false;
		}
		
		int ourFirstIndex = getFirstIndex();
		int otherFirstIndex = other.getFirstIndex();
		
		for (int i = 0; i < getSize(); i++) {
			if (getValueAt(ourFirstIndex+i) != other.getValueAt(otherFirstIndex+i)) {
				return false;
			}
		}
		
		return true;
	}

	@Override
	public String toString() {
		StringBuffer buffer = new StringBuffer(16);
		buffer.append ("(");
		
		for (int i = 1; i <= getSize(); i++) {
			buffer.append (getValueAt(i));
			if (i > 0 && i < getSize()) {
				buffer.append (",");
			}
		}
		buffer.append(")");
		return buffer.toString();
	}
	public boolean equals(Vector v) {
		if(this.getNumberOfColumns()==v.getNumberOfColumns() && this.getNumberOfRows()==v.getNumberOfRows()) {
			for(int row =0;row<this.getNumberOfRows(); row++) {
				for(int column=0;column<this.getNumberOfColumns();column++) {
					if(this.getValueAt(row, column)!=v.getValueAt(row, column)) {
						return false;
					}
				}
			}
			return true;
		}
		return false;
	}
	
}
